# =============================================================================
# Honaramiz Animation Tools
# Developed by Honaramiz Team | https://honaramiz.com    
# Features:
#   - Paste Relative
#   - Clean Static Keys
#   - Drive Animation: SDK  + Node-Based 
#   - Smart Bake: auto-detect frame range from existing animation
#   - Full curve preservation (tangents, weights, infinity)
#   - Supports Maya 2014–2025+
#===============================================================================
# How to Install & Run honaramizAnimTools in Maya
# Step 1: Save the Script File
# Place honaramizAnimTools.py file in your Maya scripts directory:
# Windows:
# C:\Users\<YourUsername>\Documents\maya\scripts
# macOS:
# /Users/<YourUsername>/Library/Preferences/Autodesk/maya/scripts
# Linux:
# /home/<YourUsername>/maya/scripts
# Replace <YourUsername> with your actual system username.
# Step 2: Create a Shelf Button 
# In the Command box, paste the following code:
# import honaramizAnimTools
# honaramizAnimTools.honaramizAnimTools()
# =============================================================================

import maya.cmds as cmds
import random
import webbrowser

def open_honaramiz_website(*args):
    webbrowser.open("https://honaramiz.com")

class honaramizAnimTools:
    def __init__(self):
        self.window = "animOffsetToolWin"
        self.copyBuffer = {}
        self.attrCopyBuffer = {}
        self.copyBufferData = {}
        self.attrCopyBufferData = {}
        self.buildUI()

    def buildUI(self):
        if cmds.window(self.window, exists=True):
            cmds.deleteUI(self.window)

        self.window = cmds.window(
            self.window,
            title="🎬 Honaramiz Animation Tools",
            sizeable=True,
            resizeToFitChildren=True
        )
        mainLayout = cmds.columnLayout(adj=True, rs=6)

        # ───────────────── OFFSET KEYS ─────────────────
        offsetFrame = cmds.frameLayout(
            label="⏯️ Offset Keys",
            collapsable=True,
            bgc=[0.18, 0.18, 0.18],
            mw=8, mh=8
        )
        cmds.columnLayout(adj=True, rs=6)

        self.offsetValueField = cmds.intFieldGrp(
            label="Offset Amount:",
            value1=5,
            cw2=[100, 140]
        )

        self.offsetTypeMenu = cmds.optionMenu(label="Offset Type:")
        cmds.menuItem(label="Constant Offset")
        cmds.menuItem(label="Random per Object")
        cmds.menuItem(label="Random per Keyframe")
        cmds.menuItem(label="Sequential per Object")

        self.preserveFirstCB = cmds.checkBox(label="Preserve First Key", value=False)

        cmds.separator(h=10)
        cmds.button(
            label="✅ Apply Offset",
            h=36,
            bgc=[0.25, 0.65, 0.35],
            c=self.applyOffset
        )
        cmds.setParent("..")
        cmds.setParent("..")

        # ───────────────── CHANNEL TOOLS (ALL ATTRIBUTES) ─────────────────
        chanFrame = cmds.frameLayout(
            label="🧰 Channel Tools ",
            collapsable=True,
            bgc=[0.20, 0.20, 0.20],
            mw=8, mh=8
        )
        cmds.columnLayout(adj=True, rs=6)
        cmds.text(label="Applies to ALL animated keyable/channel attributes", align="center", font="obliqueLabelFont")
        cmds.separator(h=8)
        cmds.rowLayout(nc=4, cw4=[85, 85, 95, 85])
        cmds.button(label="📋 Copy", h=30, w=85, c=self.copyAllAnimatedAttrs)
        cmds.button(label="📋 Paste", h=30, w=85, c=self.pasteAllAnimatedAttrs)
        cmds.button(label="📎 Paste Rel", h=30, w=95, c=self.pasteRelativeAllAttrs)
        cmds.button(label="🗑️ Delete", h=30, w=85, bgc=[0.6, 0.3, 0.3], c=self.deleteAllAnimatedAttrs)
        cmds.setParent("..")
        cmds.setParent("..")
        cmds.setParent("..")

        # ───────────────── ATTRIBUTE TOOLS ─────────────────
        attrFrame = cmds.frameLayout(
            label="🔧 Selected Attribute Tools",
            collapsable=True,
            bgc=[0.22, 0.22, 0.22],
            mw=8, mh=8
        )
        cmds.columnLayout(adj=True, rs=6)

        cmds.text(label="Applies to ALL selected objects", align="center", font="obliqueLabelFont")
        cmds.separator(h=6)

        cmds.rowLayout(nc=2, cw2=[170, 170])
        cmds.button(label="🔑 Key Attribute", h=28, c=self.keySelectedAttr)
        cmds.button(label="🗑️ Delete Keys", h=28, bgc=[0.6, 0.3, 0.3], c=self.deleteSelectedAttrKeys)
        cmds.setParent("..")

        cmds.separator(h=6)
        cmds.rowLayout(nc=3, cw3=[110, 110, 110])
        cmds.button(label="📋 Copy Attr", h=28, w=110, c=self.copySelectedAttrKeys)
        cmds.button(label="📋 Paste Attr", h=28, w=110, c=self.pasteSelectedAttrKeys)
        cmds.button(label="📎 Paste Rel", h=28, w=110, c=self.pasteRelativeSelectedAttrKeys)
        cmds.setParent("..")
        cmds.setParent("..")
        cmds.setParent("..")

        # ───────────────── VALUE OPERATIONS ─────────────────
        opFrame = cmds.frameLayout(
            label="✨ Value Operations",
            collapsable=True,
            bgc=[0.24, 0.24, 0.24],
            mw=8, mh=8
        )
        cmds.columnLayout(adj=True, rs=6)
        cmds.gridLayout(numberOfColumns=3, cellWidthHeight=(115, 28))
        cmds.button(label="🔄 Mirror Values", c=self.mirrorSelectedAttrValues)
        cmds.button(label="🎲 Randomize", c=self.randomizeSelectedAttrValues)
        cmds.button(label="✨ Smooth", c=self.smoothSelectedAttrValues)
        cmds.button(label="📈 Scale", c=self.scaleSelectedAttrValues)
        cmds.button(label="🔼 Offset", c=self.offsetSelectedAttrValues)
        cmds.button(label="🔢 Set Constant", c=self.setConstantValue)
        cmds.button(label="🔄 Invert Avg", c=self.invertAroundAverage)
        cmds.button(label="⚖️ Normalize", c=self.normalizeValues)
        cmds.button(label="🧹 Clean Static", c=self.cleanUpChannelKeys)
        cmds.setParent("..")
        cmds.setParent("..")
        cmds.setParent("..")

        # ───────────────── ANIMATION DRIVING SYSTEM ─────────────────
        driveFrame = cmds.frameLayout(
            label="🔗 Animation Driving System",
            collapsable=True,
            collapse=True,
            bgc=[0.26, 0.28, 0.35],
            mw=8, mh=8
        )
        cmds.columnLayout(adj=True, rs=8)
        cmds.text(label="Select animated objects + ONE driver object (last selected)", align="center", font="obliqueLabelFont")
        cmds.text(label="All animation will be connected to a single attribute", align="center", font="smallPlainLabelFont")
        cmds.separator(h=8)
        cmds.button(
            label="🔗 Drive with SDK",
            h=32,
            bgc=[0.4, 0.5, 0.65],
            c=self.batchDriveAnimation
        )
        cmds.button(
            label="🎯 Drive with Nodes",
            h=32,
            bgc=[0.35, 0.55, 0.75],
            c=self.batchDriveWithNodes
        )
        cmds.separator(h=6)
        cmds.button(
            label="🔥 Bake All Animated Frames",
            h=30,
            bgc=[0.6, 0.4, 0.2],
            c=self.bakeDrivenAnimation
        )
        cmds.setParent("..")
        cmds.setParent("..")

        # ───────────────── FOOTER ─────────────────
        cmds.separator(h=12)
        cmds.text(label="Developed by Honaramiz Team", align="center", font="smallObliqueLabelFont")
        cmds.button(
            label="🌐 Visit honaramiz.com",
            command=open_honaramiz_website,
            bgc=[0.15, 0.15, 0.15],
            h=26
        )

        cmds.showWindow(self.window)

    # ───────────────── UTILITIES ─────────────────
    def getAnimatedNumericAttrs(self, obj):
        """Returns all animated, numeric, scalar attributes (keyable or channelBox)."""
        keyable = cmds.listAttr(obj, keyable=True) or []
        channel = cmds.listAttr(obj, channelBox=True) or []
        all_attrs = list(set(keyable + channel))
        animated_attrs = []
        for attr in all_attrs:
            full = f"{obj}.{attr}"
            try:
                if cmds.listConnections(full, type="animCurve", source=True, destination=False):
                    val = cmds.getAttr(full)
                    if isinstance(val, (int, float, bool)) or (isinstance(val, tuple) and len(val) == 1):
                        animated_attrs.append(full)
            except:
                continue
        return animated_attrs

    def _getSelectedAttrs(self):
        attrs = cmds.channelBox('mainChannelBox', q=True, sma=True)
        if not attrs:
            cmds.warning("No attributes selected in Channel Box.")
            return None
        return attrs

    # ───────────────── CORE: APPLY OFFSET ─────────────────
    def applyOffset(self, *args):
        offset_amount = cmds.intFieldGrp(self.offsetValueField, q=True, value1=True)
        offset_type = cmds.optionMenu(self.offsetTypeMenu, q=True, v=True)
        preserve_first = cmds.checkBox(self.preserveFirstCB, q=True, v=True)
        sel = cmds.ls(sl=True)
        if not sel:
            cmds.warning("Select at least one object.")
            return

        if offset_type == "Sequential per Object":
            obj_offsets = {obj: i * offset_amount for i, obj in enumerate(sel)}
        else:
            obj_offsets = None

        for obj in sel:
            attrs = self.getAnimatedNumericAttrs(obj)
            if offset_type == "Sequential per Object":
                obj_offset = obj_offsets[obj]
            elif offset_type == "Random per Object":
                obj_offset = random.randint(-offset_amount, offset_amount)
            else:
                obj_offset = offset_amount

            for attr in attrs:
                if not cmds.objExists(attr): continue
                times = cmds.keyframe(attr, q=True, tc=True)
                if not times: continue

                if offset_type == "Random per Keyframe":
                    times = cmds.keyframe(attr, q=True, tc=True)
                    values = cmds.keyframe(attr, q=True, vc=True)
                    cmds.cutKey(attr)
                    first_t = min(times) if preserve_first else None
                    for t, v in zip(times, values):
                        if preserve_first and first_t is not None and abs(t - first_t) < 0.01:
                            new_t = t
                        else:
                            new_t = t + random.randint(-offset_amount, offset_amount)
                        cmds.setKeyframe(attr, t=new_t, v=v)
                else:
                    if preserve_first:
                        first_t = min(times)
                        cmds.keyframe(attr, edit=True, relative=True, timeChange=obj_offset, t=(first_t + 0.001, max(times) + 100000))
                    else:
                        cmds.keyframe(attr, edit=True, relative=True, timeChange=obj_offset)

        cmds.inViewMessage(amg=f'<b>✅ Offset Applied</b><br>Type: {offset_type}', pos='midCenter', fade=True)

    # ───────────────── ALL ATTRIBUTE CLONING ─────────────────
    def copyAllAnimatedAttrs(self, *args):
        sel = cmds.ls(sl=True)
        if not sel:
            cmds.warning("Select object(s).")
            return
        self.copyBuffer = {}
        self.copyBufferData = {}
        obj = sel[0]
        for attr in self.getAnimatedNumericAttrs(obj):
            base = attr.split(".")[-1]
            curves = cmds.listConnections(attr, type="animCurve", source=True, destination=False)
            if curves:
                curve = curves[0]
                self.copyBuffer[base] = curve
                times = cmds.keyframe(curve, q=True, tc=True) or []
                values = cmds.keyframe(curve, q=True, vc=True) or []
                if times and values:
                    self.copyBufferData[base] = {
                        'times': times,
                        'values': values,
                        'reference': values[0]
                    }
        cmds.inViewMessage(amg='<b>📋 All Animated Attributes Copied</b>', pos='midCenter', fade=True)

    def pasteAllAnimatedAttrs(self, *args):
        if not self.copyBuffer:
            cmds.warning("Nothing to paste.")
            return
        sel = cmds.ls(sl=True)
        if not sel:
            cmds.warning("Select target object(s).")
            return

        for obj in sel:
            for base, src_curve in self.copyBuffer.items():
                target_attr = obj + "." + base
                if not cmds.attributeQuery(base, node=obj, exists=True):
                    continue

                old_curves = cmds.listConnections(target_attr, type="animCurve", source=True, destination=False)
                if old_curves:
                    cmds.delete(old_curves)

                try:
                    new_curve = cmds.duplicate(src_curve, name=obj + "_" + base + "_anim")[0]
                    cmds.connectAttr(new_curve + ".output", target_attr, force=True)
                except Exception as e:
                    cmds.warning(f"Failed to paste curve for {target_attr}: {e}")

        cmds.inViewMessage(amg='<b>📋 All Animated Attributes Pasted</b>', pos='midCenter', fade=True)

    def pasteRelativeAllAttrs(self, *args):
        if not self.copyBufferData:
            cmds.warning("No relative data to paste.")
            return
        sel = cmds.ls(sl=True)
        if not sel:
            cmds.warning("Select target object(s).")
            return

        for obj in sel:
            for base, data in self.copyBufferData.items():
                target_attr = obj + "." + base
                if not cmds.attributeQuery(base, node=obj, exists=True):
                    continue

                ref_pasted = data['reference']
                dest_times = cmds.keyframe(target_attr, q=True, tc=True)
                dest_values = cmds.keyframe(target_attr, q=True, vc=True)
                if not dest_times or not dest_values:
                    current_val = cmds.getAttr(target_attr)
                    dest_times = [cmds.currentTime(q=True)]
                    dest_values = [current_val]

                current_dict = dict(zip(dest_times, dest_values))
                new_keys = {}
                for t, v in zip(data['times'], data['values']):
                    offset = v - ref_pasted
                    current_val_at_t = current_dict.get(t, cmds.getAttr(target_attr))
                    new_val = current_val_at_t + offset
                    new_keys[t] = new_val

                old_curves = cmds.listConnections(target_attr, type="animCurve", source=True, destination=False)
                if old_curves:
                    cmds.delete(old_curves)

                for t, v in new_keys.items():
                    cmds.setKeyframe(target_attr, t=t, v=v)

        cmds.inViewMessage(amg='<b>📎 Relative Paste Applied to All Attributes</b>', pos='midCenter', fade=True)

    def deleteAllAnimatedAttrs(self, *args):
        sel = cmds.ls(sl=True)
        if not sel: return
        for obj in sel:
            for attr in self.getAnimatedNumericAttrs(obj):
                old_curves = cmds.listConnections(attr, type="animCurve", source=True, destination=False)
                if old_curves:
                    cmds.delete(old_curves)
        cmds.inViewMessage(amg='<b>🗑️ All Animated Attributes Deleted</b>', pos='midCenter', fade=True)

    # ───────────────── ATTRIBUTE CURVE CLONING (Selected in Channel Box) ─────────────────
    def keySelectedAttr(self, *args):
        sel = cmds.ls(sl=True)
        if not sel:
            cmds.warning("Select at least one object.")
            return
        attrs = self._getSelectedAttrs()
        if not attrs: return
        for obj in sel:
            for a in attrs:
                full = obj + "." + a
                if cmds.attributeQuery(a, node=obj, exists=True):
                    cmds.setKeyframe(full)
        cmds.inViewMessage(amg=f'<b>🔑 Keyed {len(attrs)} Attributes</b>', pos='midCenter', fade=True)

    def deleteSelectedAttrKeys(self, *args):
        sel = cmds.ls(sl=True)
        if not sel:
            cmds.warning("Select at least one object.")
            return
        attrs = self._getSelectedAttrs()
        if not attrs: return
        for obj in sel:
            for a in attrs:
                full = obj + "." + a
                if cmds.attributeQuery(a, node=obj, exists=True):
                    old_curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                    if old_curves:
                        cmds.delete(old_curves)
        cmds.inViewMessage(amg=f'<b>🗑️ Curves Deleted for {len(attrs)} Attributes</b>', pos='midCenter', fade=True)

    def copySelectedAttrKeys(self, *args):
        sel = cmds.ls(sl=True)
        if not sel:
            cmds.warning("Select at least one object.")
            return
        attrs = self._getSelectedAttrs()
        if not attrs: return
        obj = sel[0]
        self.attrCopyBuffer = {}
        self.attrCopyBufferData = {}
        for a in attrs:
            full = obj + "." + a
            if not cmds.attributeQuery(a, node=obj, exists=True): continue
            curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
            if curves:
                curve = curves[0]
                self.attrCopyBuffer[a] = curve
                times = cmds.keyframe(curve, q=True, tc=True) or []
                values = cmds.keyframe(curve, q=True, vc=True) or []
                if times and values:
                    self.attrCopyBufferData[a] = {
                        'times': times,
                        'values': values,
                        'reference': values[0]
                    }
        cmds.inViewMessage(amg=f'<b>📋 Copied Curves & Data from {obj}</b>', pos='midCenter', fade=True)

    def pasteSelectedAttrKeys(self, *args):
        if not self.attrCopyBuffer:
            cmds.warning("Nothing to paste.")
            return
        sel = cmds.ls(sl=True)
        if not sel:
            cmds.warning("Select target object(s).")
            return

        for obj in sel:
            for a, src_curve in self.attrCopyBuffer.items():
                full = obj + "." + a
                if not cmds.attributeQuery(a, node=obj, exists=True): continue

                old_curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                if old_curves:
                    cmds.delete(old_curves)

                try:
                    new_curve = cmds.duplicate(src_curve, name=obj + "_" + a + "_anim")[0]
                    cmds.connectAttr(new_curve + ".output", full, force=True)
                except Exception as e:
                    cmds.warning(f"Failed to paste curve for {a} on {obj}: {e}")

        cmds.inViewMessage(amg='<b>📋 Attribute Curves Cloned</b>', pos='midCenter', fade=True)

    def pasteRelativeSelectedAttrKeys(self, *args):
        if not self.attrCopyBufferData:
            cmds.warning("No relative attribute data to paste.")
            return
        sel = cmds.ls(sl=True)
        if not sel:
            cmds.warning("Select target object(s).")
            return
        attrs = self._getSelectedAttrs()
        if not attrs: return

        for obj in sel:
            for a, data in self.attrCopyBufferData.items():
                if a not in attrs:
                    continue
                full = obj + "." + a
                if not cmds.attributeQuery(a, node=obj, exists=True):
                    continue

                ref_pasted = data['reference']
                dest_times = cmds.keyframe(full, q=True, tc=True)
                dest_values = cmds.keyframe(full, q=True, vc=True)
                if not dest_times or not dest_values:
                    current_val = cmds.getAttr(full)
                    dest_times = [cmds.currentTime(q=True)]
                    dest_values = [current_val]

                current_dict = dict(zip(dest_times, dest_values))
                new_keys = {}
                for t, v in zip(data['times'], data['values']):
                    offset = v - ref_pasted
                    current_val_at_t = current_dict.get(t, cmds.getAttr(full))
                    new_val = current_val_at_t + offset
                    new_keys[t] = new_val

                old_curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                if old_curves:
                    cmds.delete(old_curves)

                for t, v in new_keys.items():
                    cmds.setKeyframe(full, t=t, v=v)

        cmds.inViewMessage(amg='<b>📎 Relative Attribute Paste Applied</b>', pos='midCenter', fade=True)

    # ───────────────── VALUE OPERATIONS ─────────────────
    def _modifyValuesPreserveCurve(self, modifier_func, operation_name="Operation"):
        sel = cmds.ls(sl=True)
        attrs = self._getSelectedAttrs()
        if not sel or not attrs:
            return

        for obj in sel:
            for a in attrs:
                full = obj + "." + a
                if not cmds.attributeQuery(a, node=obj, exists=True):
                    continue

                curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                if not curves:
                    continue
                curve = curves[0]

                num_keys = cmds.keyframe(curve, q=True, keyframeCount=True)
                if not num_keys:
                    continue

                for i in range(num_keys):
                    current_val = cmds.keyframe(curve, index=(i,), q=True, vc=True)[0]
                    new_val = modifier_func(current_val)
                    cmds.keyframe(curve, index=(i,), valueChange=new_val)

        cmds.inViewMessage(amg=f'<b>✨ {operation_name} Applied (Curves Preserved)</b>', pos='midCenter', fade=True)

    def mirrorSelectedAttrValues(self, *args):
        self._modifyValuesPreserveCurve(lambda v: -v, "Mirror Values")

    def randomizeSelectedAttrValues(self, *args):
        off = float(cmds.intFieldGrp(self.offsetValueField, q=True, value1=True))
        self._modifyValuesPreserveCurve(lambda v: v + random.uniform(-off, off), "Randomize Values")

    def smoothSelectedAttrValues(self, *args):
        sel = cmds.ls(sl=True)
        attrs = self._getSelectedAttrs()
        if not sel or not attrs:
            return

        for obj in sel:
            for a in attrs:
                full = obj + "." + a
                if not cmds.attributeQuery(a, node=obj, exists=True):
                    continue
                curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                if not curves:
                    continue
                curve = curves[0]
                num_keys = cmds.keyframe(curve, q=True, keyframeCount=True)
                if num_keys < 3:
                    continue

                values = [cmds.keyframe(curve, index=(i,), q=True, vc=True)[0] for i in range(num_keys)]
                smoothed = [
                    values[i] if i == 0 or i == num_keys - 1
                    else (values[i-1] + values[i] + values[i+1]) / 3.0
                    for i in range(num_keys)
                ]

                for i, v in enumerate(smoothed):
                    cmds.keyframe(curve, index=(i,), valueChange=v)

        cmds.inViewMessage(amg='<b>✨ Values Smoothed (Curves Preserved)</b>', pos='midCenter', fade=True)

    def scaleSelectedAttrValues(self, *args):
        result = cmds.promptDialog(
            title='Scale Values',
            message='Enter scale factor (e.g. 1.5):',
            button=['OK', 'Cancel'],
            defaultButton='OK'
        )
        if result != 'OK':
            return
        try:
            factor = float(cmds.promptDialog(q=True, text=True))
        except:
            cmds.warning("Invalid number.")
            return
        self._modifyValuesPreserveCurve(lambda v: v * factor, f"Scale by {factor}x")

    def offsetSelectedAttrValues(self, *args):
        result = cmds.promptDialog(
            title='Offset Values',
            message='Enter offset amount (e.g. 5):',
            button=['OK', 'Cancel'],
            defaultButton='OK'
        )
        if result != 'OK':
            return
        try:
            offset_val = float(cmds.promptDialog(q=True, text=True))
        except:
            cmds.warning("Invalid number.")
            return
        self._modifyValuesPreserveCurve(lambda v: v + offset_val, f"Offset by {offset_val}")

    def setConstantValue(self, *args):
        result = cmds.promptDialog(
            title='Set Constant Value',
            message='Enter constant value (e.g. 0):',
            button=['OK', 'Cancel'],
            defaultButton='OK'
        )
        if result != 'OK':
            return
        try:
            const_val = float(cmds.promptDialog(q=True, text=True))
        except:
            cmds.warning("Invalid number.")
            return

        sel = cmds.ls(sl=True)
        attrs = self._getSelectedAttrs()
        if not sel or not attrs:
            return

        for obj in sel:
            for a in attrs:
                full = obj + "." + a
                if not cmds.attributeQuery(a, node=obj, exists=True):
                    continue
                curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                if not curves:
                    continue
                curve = curves[0]
                num_keys = cmds.keyframe(curve, q=True, keyframeCount=True)
                for i in range(num_keys):
                    cmds.keyframe(curve, index=(i,), valueChange=const_val)

        cmds.inViewMessage(amg=f'<b>🔢 All Keys Set to {const_val}</b>', pos='midCenter', fade=True)

    def invertAroundAverage(self, *args):
        sel = cmds.ls(sl=True)
        attrs = self._getSelectedAttrs()
        if not sel or not attrs:
            return

        for obj in sel:
            for a in attrs:
                full = obj + "." + a
                if not cmds.attributeQuery(a, node=obj, exists=True):
                    continue
                curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                if not curves:
                    continue
                curve = curves[0]
                num_keys = cmds.keyframe(curve, q=True, keyframeCount=True)
                if not num_keys:
                    continue

                values = [cmds.keyframe(curve, index=(i,), q=True, vc=True)[0] for i in range(num_keys)]
                avg = sum(values) / len(values)
                for i, v in enumerate(values):
                    new_v = 2 * avg - v
                    cmds.keyframe(curve, index=(i,), valueChange=new_v)

        cmds.inViewMessage(amg='<b>🔄 Inverted Around Average</b>', pos='midCenter', fade=True)

    def normalizeValues(self, *args):
        mode = cmds.confirmDialog(
            title='Normalize Range',
            message='Choose range:',
            button=['0 to 1', '-1 to 1', 'Cancel'],
            defaultButton='-1 to 1'
        )
        if mode == 'Cancel':
            return

        sel = cmds.ls(sl=True)
        attrs = self._getSelectedAttrs()
        if not sel or not attrs:
            return

        for obj in sel:
            for a in attrs:
                full = obj + "." + a
                if not cmds.attributeQuery(a, node=obj, exists=True):
                    continue
                curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                if not curves:
                    continue
                curve = curves[0]
                num_keys = cmds.keyframe(curve, q=True, keyframeCount=True)
                if not num_keys:
                    continue

                values = [cmds.keyframe(curve, index=(i,), q=True, vc=True)[0] for i in range(num_keys)]
                min_v, max_v = min(values), max(values)
                if min_v == max_v:
                    continue

                if mode == '0 to 1':
                    new_values = [(v - min_v) / (max_v - min_v) for v in values]
                else:
                    mid = (min_v + max_v) / 2.0
                    amp = (max_v - min_v) / 2.0
                    new_values = [(v - mid) / amp for v in values]

                for i, nv in enumerate(new_values):
                    cmds.keyframe(curve, index=(i,), valueChange=nv)

        cmds.inViewMessage(amg=f'<b>⚖️ Normalized to {mode}</b>', pos='midCenter', fade=True)

    def cleanUpChannelKeys(self, *args):
        sel = cmds.ls(sl=True)
        attrs = self._getSelectedAttrs()
        if not sel or not attrs:
            return

        cleaned_count = 0
        for obj in sel:
            for a in attrs:
                full = obj + "." + a
                if not cmds.attributeQuery(a, node=obj, exists=True):
                    continue

                curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                if not curves:
                    continue
                curve = curves[0]

                num_keys = cmds.keyframe(curve, q=True, keyframeCount=True)
                if not num_keys or num_keys < 2:
                    continue

                values = [cmds.keyframe(curve, index=(i,), q=True, vc=True)[0] for i in range(num_keys)]
                first_val = values[0]

                if all(abs(v - first_val) < 1e-5 for v in values):
                    current_val = cmds.getAttr(full)
                    if abs(current_val - first_val) < 1e-5:
                        cmds.delete(curve)
                        cleaned_count += 1

        if cleaned_count > 0:
            cmds.inViewMessage(amg=f'<b>🧹 {cleaned_count} Static Channel(s) Cleaned</b>', pos='midCenter', fade=True)
        else:
            cmds.inViewMessage(amg='<b>ℹ️ No Static Channels Found</b>', pos='midCenter', fade=True)

    # ───────────────── BATCH DRIVING — SDK METHOD  ─────────────────
    def batchDriveAnimation(self, *args):
        sel = cmds.ls(sl=True)
        if len(sel) < 2:
            cmds.warning("Select animated objects + ONE driver object (last selected).")
            return

        if cmds.window("sdkDriveOpts", exists=True):
            cmds.deleteUI("sdkDriveOpts")

        driver_obj = sel[-1]

        opts_win = cmds.window("sdkDriveOpts", title="SDK Drive Options", sizeable=False)
        cmds.columnLayout(adj=True, rs=8, cat=["both", 12])
        cmds.text(label="Driver Object: " + driver_obj, align="left", font="boldLabelFont")
        self.driveAttrFieldSDK = cmds.textFieldGrp(label="Driver Attribute:", text="mainDrive", columnWidth2=[120, 180])
        self.normalizeTimingSDK = cmds.checkBox(label="Use Normalized Time (0→1)", value=False, align="left")
        cmds.separator(h=15)
        cmds.rowLayout(nc=2, cw2=[150, 150])
        cmds.button(label="✅ Apply", h=30, bgc=[0.3, 0.6, 0.4], c=self._executeSDKDrive)
        cmds.button(label="❌ Cancel", h=30, bgc=[0.5, 0.3, 0.3], c=lambda *_: cmds.deleteUI("sdkDriveOpts"))
        cmds.setParent("..")
        cmds.showWindow(opts_win)

    def _executeSDKDrive(self, *args):
        try:
            driver_attr = cmds.textFieldGrp(self.driveAttrFieldSDK, q=True, text=True).strip()
            normalize = cmds.checkBox(self.normalizeTimingSDK, q=True, v=True)
        except Exception as e:
            cmds.warning("Failed to read UI values: " + str(e))
            return

        if cmds.window("sdkDriveOpts", exists=True):
            cmds.deleteUI("sdkDriveOpts")

        if not driver_attr:
            cmds.warning("Driver attribute name is empty.")
            return

        sel = cmds.ls(sl=True)
        if len(sel) < 2:
            cmds.warning("Select animated objects + ONE driver object.")
            return

        driver_obj = sel[-1]
        driven_objs = sel[:-1]
        full_driver_attr = f"{driver_obj}.{driver_attr}"

        if not cmds.attributeQuery(driver_attr, node=driver_obj, exists=True):
            cmds.addAttr(driver_obj, longName=driver_attr, attributeType='double', defaultValue=0, keyable=True)

        processed = 0
        for obj in driven_objs:
            keyed_attrs = self.getAnimatedNumericAttrs(obj)

            for attr in keyed_attrs:
                times = cmds.keyframe(attr, q=True, tc=True)
                values = cmds.keyframe(attr, q=True, vc=True)
                if not times or not values:
                    continue

                curves = cmds.listConnections(attr, type="animCurve", source=True, destination=False)
                if curves:
                    cmds.delete(curves)

                if normalize:
                    t_min, t_max = min(times), max(times)
                    if t_max == t_min:
                        driver_vals = [0.0] * len(times)
                    else:
                        driver_vals = [(t - t_min) / (t_max - t_min) for t in times]
                else:
                    driver_vals = times

                for drv_val, v in zip(driver_vals, values):
                    cmds.setDrivenKeyframe(
                        attr,
                        currentDriver=full_driver_attr,
                        driverValue=drv_val,
                        value=v
                    )
                processed += 1

        if processed > 0:
            mode = " (Normalized)" if normalize else " (Frame-Based)"
            cmds.inViewMessage(
                amg=f'<b>🔗 {processed} Attributes Driven (SDK)!{mode}</b><br>Driver: {full_driver_attr}<br>• All animatable channel box attrs',
                pos='midCenter',
                fade=True
            )
        else:
            cmds.warning("No animated attributes found.")

    # ───────────────── BATCH DRIVING — NODE-BASED  ─────────────────
    def batchDriveWithNodes(self, *args):
        sel = cmds.ls(sl=True)
        if len(sel) < 2:
            cmds.warning("Select animated objects + ONE driver object (last selected).")
            return

        if cmds.window("nodeDriveOpts", exists=True):
            cmds.deleteUI("nodeDriveOpts")

        driver_obj = sel[-1]

        opts_win = cmds.window("nodeDriveOpts", title="Node Drive Options", sizeable=False)
        cmds.columnLayout(adj=True, rs=8, cat=["both", 12])
        cmds.text(label="Driver Object: " + driver_obj, align="left", font="boldLabelFont")
        self.driveAttrFieldND = cmds.textFieldGrp(label="Driver Attribute:", text="mainDrive", columnWidth2=[120, 180])
        self.normalizeTiming = cmds.checkBox(label="Use Normalized Time (0→1)", value=False, align="left")
        cmds.separator(h=15)
        cmds.rowLayout(nc=2, cw2=[150, 150])
        cmds.button(label="✅ Apply", h=30, bgc=[0.3, 0.6, 0.4], c=self._executeNodeDrive)
        cmds.button(label="❌ Cancel", h=30, bgc=[0.5, 0.3, 0.3], c=lambda *_: cmds.deleteUI("nodeDriveOpts"))
        cmds.setParent("..")
        cmds.showWindow(opts_win)

    def _executeNodeDrive(self, *args):
        try:
            driver_attr = cmds.textFieldGrp(self.driveAttrFieldND, q=True, text=True).strip()
            normalize = cmds.checkBox(self.normalizeTiming, q=True, v=True)
        except Exception as e:
            cmds.warning("Failed to read UI values: " + str(e))
            return

        if cmds.window("nodeDriveOpts", exists=True):
            cmds.deleteUI("nodeDriveOpts")

        if not driver_attr:
            cmds.warning("Driver attribute name is empty.")
            return

        sel = cmds.ls(sl=True)
        if len(sel) < 2:
            cmds.warning("Select animated objects + ONE driver object.")
            return

        driver_obj = sel[-1]
        driven_objs = sel[:-1]
        full_driver_attr = f"{driver_obj}.{driver_attr}"

        if not cmds.attributeQuery(driver_attr, node=driver_obj, exists=True):
            cmds.addAttr(driver_obj, longName=driver_attr, attributeType='double', defaultValue=0, keyable=True)

        processed = 0
        for obj in driven_objs:
            all_attrs = cmds.listAttr(obj, keyable=True, visible=True, scalar=True) or []
            for a in all_attrs:
                full = obj + "." + a
                curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                if not curves:
                    continue
                orig_curve = curves[0]

                times = cmds.keyframe(orig_curve, q=True, tc=True)
                values = cmds.keyframe(orig_curve, q=True, vc=True)
                if not times or not values:
                    continue

                cmds.disconnectAttr(orig_curve + ".output", full)

                if normalize:
                    t_min, t_max = min(times), max(times)
                    if t_max == t_min:
                        normalized_times = [0.0] * len(times)
                    else:
                        normalized_times = [(t - t_min) / (t_max - t_min) for t in times]
                else:
                    normalized_times = times

                new_curve = cmds.createNode("animCurveTU", name=orig_curve + "_driven")
                for t_norm, v in zip(normalized_times, values):
                    cmds.setKeyframe(new_curve, t=float(t_norm), value=v, inTangentType='auto', outTangentType='auto')

                cmds.connectAttr(full_driver_attr, new_curve + ".input")
                cmds.connectAttr(new_curve + ".output", full)

                processed += 1

        if processed > 0:
            mode = " (Normalized)" if normalize else " (Frame-Based)"
            cmds.inViewMessage(
                amg=f'<b>🎯 {processed} Attributes Driven!{mode}</b><br>Driver: {full_driver_attr}<br>• Animatable (0→1 or 1→0)<br>• Full curve preserved',
                pos='midCenter',
                fade=True
            )
        else:
            cmds.warning("No animated attributes found.")

    # ───────────────── BAKE ANIMATION — SMART AUTO RANGE ─────────────────
    def bakeDrivenAnimation(self, *args):
        sel = cmds.ls(sl=True)
        if not sel:
            cmds.warning("Select at least one object to bake animation.")
            return

        all_times = set()
        animated_attrs = []

        for obj in sel:
            attrs = cmds.listAttr(obj, keyable=True, visible=True, scalar=True) or []
            for a in attrs:
                full = obj + "." + a
                curves = cmds.listConnections(full, type="animCurve", source=True, destination=False)
                if curves:
                    animated_attrs.append(full)
                    times = cmds.keyframe(curves[0], q=True, tc=True)
                    if times:
                        all_times.update(times)

        if not animated_attrs:
            cmds.warning("No animated attributes found on selected objects.")
            return

        min_time = min(all_times)
        max_time = max(all_times)

        cmds.bakeResults(
            animated_attrs,
            time=(min_time, max_time),
            simulation=True,
            sampleBy=1,
            preserveOutsideKeys=True,
            sparseAnimCurveBake=False,
            removeBakedAttributeFromLayer=False,
            bakeOnOverrideLayer=False,
        )

        cmds.inViewMessage(
            amg=f'<b>🔥 Baked {len(animated_attrs)} Attributes!</b><br>Range: {int(min_time)} → {int(max_time)}',
            pos='midCenter',
            fade=True
        )

# =============================================================================
# RUN
# =============================================================================